
#ifndef CompleteSystem_h
#define CompleteSystem_h
#include "square_container.h"
#include <vector>
#include <limits>
#include <thread>
#include <mutex>
std::mutex lock;


//size parameters are stored in template variables to use std::arrays to speed up runtime
template <size_t hight,size_t width>
/// system class used to perform the simulation
class CompleteSystem {
    
public:
    
    /// Constructor for the system class
    /// @param inactive_monomers std::array with the initital number of all inactive particle species
    /// @param monomers std::array with the initital number of all active particle species
    /// @param critical_size minimal size for a seed to be stable
    /// @param activation_rate rate at which inactive monomers get activated
    /// @param decay_rate rate at which too small seeds decay
    /// @param seed seed for the random number generator (has nothing to do with the decay of assembled strucxtures!!!)
    CompleteSystem(std::array<std::array<long,width>,hight>& inactive_monomers,std::array<std::array<long,width>,hight>& monomers,int critical_size,double activation_rate,double decay_rate,unsigned seed):
    inactive_monomers_{inactive_monomers},
    monomers_{monomers},
    rng_engine_{seed},
    critical_size_{critical_size},
    finished_squares_{0},
    all_possible_monomer_bindings_{0},
    all_inactive_monomers_{0},
    activation_rate_{activation_rate},
    decay_rate_{decay_rate},
    time_{0},
    max_number_of_squares_{std::numeric_limits<long>::max()},
    accumulated_rates_(2.0,0.0){
        
        //sets up all rates
        CountPossibleMonomerBindings();
        
        //counts all iniactive monomers
        for (auto& x: inactive_monomers_) {
            for(auto& y: x){
                all_inactive_monomers_+=y;
            }
        }
        
        //max number of possible squares for the definition of the yield
        for (int i=0; i<hight; ++i) {
            for (int j=0; j<width; ++j) {
                if (monomers_[i][j]+inactive_monomers_[i][j]<max_number_of_squares_) {
                    max_number_of_squares_=monomers_[i][j]+inactive_monomers_[i][j];
                }
            }
        }
    }
    
    
    /// Copy contructor
    /// @param other_sys system to copy
    /// @param seed seed for the random number generater for different time evolution
    CompleteSystem(CompleteSystem* other_sys,unsigned seed):
    inactive_monomers_{other_sys->inactive_monomers_},
    monomers_{other_sys->monomers_},
    rng_engine_{seed},
    time_{other_sys->time_},
    critical_size_{other_sys->critical_size_},
    finished_squares_{other_sys->finished_squares_},
    all_possible_monomer_bindings_{other_sys->all_possible_monomer_bindings_},
    activation_rate_{other_sys->activation_rate_},
    decay_rate_{other_sys->decay_rate_},
    max_number_of_squares_{other_sys->max_number_of_squares_},
    all_inactive_monomers_{other_sys->all_inactive_monomers_},
    accumulated_rates_(other_sys->accumulated_rates_){
    }
    
    
    /// Count number of binding partners for a monomer
    /// @param row row of the monomer
    /// @param column column of the monomer
    /// @return number of possible finding partners of a monomer (direction per definition left and below)
    const long MonomerGetBindingPartners(const int row,const int column)const{
            long number=0;
            if (row) {
                number+=monomers_[row-1][column];
            }
            if (column) {
                number+=monomers_[row][column-1];
            }
            return number;
    }
    
    /// Counts all binding possibilities for each monomer species
    void CountPossibleMonomerBindings(){
        all_possible_monomer_bindings_=0;
        for (int i=0; i<hight; ++i) {
            for (int j=0; j<width; ++j) {
                all_possible_monomer_bindings_+=monomers_[i][j]*MonomerGetBindingPartners(i, j);
            }
        }
    }
    
    /// Remove an active monomer
    /// @param row row of the monomer
    /// @param column column of the monomer
    void RemoveMonomer(const int row,const int column){
        --monomers_[row][column];
        all_possible_monomer_bindings_-=MonomerGetBindingPartners(row, column);
        if (row!=hight-1) {
            all_possible_monomer_bindings_-=monomers_[row+1][column];
        }
        if (column!=width-1) {
            all_possible_monomer_bindings_-=monomers_[row][column+1];
        }
    }
    
    
    /// Assemble a square from two monomers
    /// @param row1 row of monomer 1
    /// @param column1 column of monomer 1
    /// @param row2 row of monomer 2
    /// @param column2 column of monomer 2
    void CreateSquare(const int row1,const int column1,const int row2,const int column2){
        RemoveMonomer(row1, column1);
        RemoveMonomer(row2, column2);
        squares_.push_back(Square<hight, width>(SquareElement{row1,column1},SquareElement{row2,column2}));
        accumulated_rates_.push_back(0.0);
    }
    
    
    /// Calculate the rates for the next Gillespie Step
    void MakeRates(){
        accumulated_rates_[0]=double(all_inactive_monomers_)*activation_rate_;
        accumulated_rates_[1]=accumulated_rates_[0]+double(all_possible_monomer_bindings_);
        for (long i=0; i<squares_.size(); ++i) {
            accumulated_rates_[i+2]=accumulated_rates_[i+1]+squares_[i].GetRate(monomers_,critical_size_,decay_rate_);
        }
        
    }
    
    /// Remove a square from the vector of all squares currently assembled
    /// @param index index of the square in the vector
    void RemoveSquareFromVector(const long index){
        if (squares_.size()-1==index) {
            squares_.pop_back();
        }
        else{
            squares_[index]=squares_.back();
            squares_.pop_back();
        }
        accumulated_rates_.pop_back();
    }
    
    /// Activate an inactive monomer
    void ActivateMonomer(){
        long random_number=std::uniform_int_distribution<long>(1,all_inactive_monomers_)(rng_engine_);
        long sum=0;
        int row=0;
        int column=0;
        
        for (; sum<random_number; ++row) {
            for (column=0; sum<random_number&&column<width; ++column) {
                sum+=inactive_monomers_[row][column];
            }
        }
        --row;
        --column;
        
        --all_inactive_monomers_;
        --inactive_monomers_[row][column];
        ++monomers_[row][column];
        all_possible_monomer_bindings_+=MonomerGetBindingPartners(row, column);
        
        if (row!=hight-1) {
            all_possible_monomer_bindings_+=monomers_[row+1][column];
        }
        if (column!=width-1) {
            all_possible_monomer_bindings_+=monomers_[row][column+1];
        }
        
    }
    
    
    /// Combine two random active monomers
    void CombineTwoMonomers(){
        long random_number=std::uniform_int_distribution<long>(1,all_possible_monomer_bindings_)(rng_engine_);
        long sum=0;
        int row=0;
        int column=0;
        for (; sum<random_number; ++row) {
            for (column=0; sum<random_number&&column<width; ++column) {
                sum+=monomers_[row][column]*MonomerGetBindingPartners(row, column);
            }
        }
        --row;
        --column;
        
        if (!row) {
            CreateSquare(row, column, row, column-1);
        }
        else if (!column){
            CreateSquare(row, column, row-1, column);
        }
        else{
            if (monomers_[row-1][column]<std::uniform_int_distribution<long>(1,monomers_[row-1][column]+monomers_[row][column-1])(rng_engine_)) {
                CreateSquare(row, column, row, column-1);
            }
            else{
                CreateSquare(row, column, row-1, column);
            }
        }
        
        
    }
    
    
    /// Increase the size of a square
    /// @param square_index index of the square in the vector of squares currently assembled
    void SquareReact(const long square_index){
        bool square_was_not_finished=true;
        SquareElement monomer;
        if (squares_[square_index].EventTakePlace(monomers_,critical_size_,decay_rate_,rng_engine_,finished_squares_,square_was_not_finished,monomer)) {
            if (square_was_not_finished) {
                CountPossibleMonomerBindings();
            }
            else{
                RemoveMonomer(monomer.row, monomer.column);
            }
            RemoveSquareFromVector(square_index);
        }
        else{
            RemoveMonomer(monomer.row, monomer.column);
        }
    }
    
    /// Choose an event for the next Gillespie step
    void ChooseEvent(){
        double random_number=std::uniform_real_distribution<double>(0,accumulated_rates_.back())(rng_engine_);
        long even_index=0;
        while (accumulated_rates_[even_index]<random_number) {
            ++even_index;
        }
        switch (even_index) {
            case 0:
                ActivateMonomer();
                break;
            case 1:
                CombineTwoMonomers();
                break;
            default:
                SquareReact(even_index-2);
                break;
        }
    }
    
    /// Perform an Gillespie iteration step
    /// @return true if the simulation reached the final state, else false
    const bool IterationStep(){
        MakeRates();
        if (accumulated_rates_.back()==0) {
            return true;
        }
        time_+=std::exponential_distribution<double>(accumulated_rates_.back())(rng_engine_);
        ChooseEvent();
        return false;
    }
    
    /// Perform the Gillespie Simulation
    /// @param time return final simulation time
    /// @param yield return final yield
    void Simulate(double& time,double& yield){
        for (unsigned i=0; ~0u; ++i) {
            if (IterationStep()) {
                break;
            }
        }
        time=time_;
        yield=double(finished_squares_)/double(max_number_of_squares_);
    }
    
    /// Perform a simulation of a copy of the system
    /// @param time time results added to this pointer
    /// @param yield yield results added to this pointer
    void operator()(unsigned seed, double* time, double* yield){
        CompleteSystem<hight,width> copy(this,seed);
        double tmp_time=0;
        double tmp_yield=0;
        copy.Simulate(tmp_time, tmp_yield);
        lock.lock();
        *time+=tmp_time;
        *yield+=tmp_yield;
        lock.unlock();
    }
    
    
    /// Perform an esnemble of Gillespie simulations
    /// @param number_of_ensembles number of ensembles performed
    /// @param time return of final results for the av. time to this pointer
    /// @param yield return of final results for the av.yield to this pointer
    void GetEnsAv(int number_of_ensembles,double* time, double* yield){
        std::vector<std::thread> threads;
        auto random_init=std::bind(std::uniform_int_distribution<unsigned>(0,~(0u)), std::ref(rng_engine_));
        for (int i=0; i<number_of_ensembles; ++i) {
            threads.push_back(std::thread{*this,random_init(), time, yield});
        }
        for (std::thread& t:threads) {
            if (t.joinable()) {
                t.join();
            }
        }
        *time/=double(number_of_ensembles);
        *yield/=double(number_of_ensembles);
    }
    

private:
    //data members of the system
    
    const int critical_size_;
    const double activation_rate_;
    const double decay_rate_;
    double time_;
    long finished_squares_;
    long all_possible_monomer_bindings_;
    long all_inactive_monomers_;
    long max_number_of_squares_;
    
    std::vector<double> accumulated_rates_;
    std::vector<Square<hight, width>> squares_;
    std::array<std::array<long,width>,hight> monomers_;
    std::array<std::array<long,width>,hight> inactive_monomers_;
    
    //rng engine
    std::mt19937 rng_engine_;
};

#endif /* CompleteSystem_h */
